<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use app\models\Request;
use app\models\User;
use app\models\Game;
use app\models\GameSession;
use app\models\Review;
use yii\data\ActiveDataProvider;

/**
 * AdminController реализует панель администратора
 */
class AdminController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity->isAdmin();
                        },
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    Yii::$app->session->setFlash('error', 'У вас нет доступа к этой странице');
                    return $this->goHome();
                },
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete-request' => ['POST'],
                    'update-request-status' => ['POST'],
                    'delete-user' => ['POST'],
                    'delete-game' => ['POST'],
                    'delete-session' => ['POST'],
                    'approve-review' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Главная страница админ-панели
     */
    public function actionIndex()
    {
        // Статистика
        $totalUsers = User::find()->count();
        $totalGames = Game::find()->count();
        $totalRequests = Request::find()->count();
        $newRequests = Request::find()->where(['status' => Request::STATUS_NEW])->count();
        $totalSessions = GameSession::find()->where(['>=', 'date', date('Y-m-d')])->count();
        $totalReviews = Review::find()->count();

        return $this->render('index', [
            'totalUsers' => $totalUsers,
            'totalGames' => $totalGames,
            'totalRequests' => $totalRequests,
            'newRequests' => $newRequests,
            'totalSessions' => $totalSessions,
            'totalReviews' => $totalReviews,
        ]);
    }

    /**
     * Управление заявками
     */
    public function actionRequests()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Request::find()
                ->with(['user', 'session.game'])
                ->orderBy(['created_at' => SORT_DESC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('requests', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Просмотр заявки
     */
    public function actionViewRequest($id)
    {
        $model = Request::findOne($id);
        
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Заявка не найдена');
        }

        return $this->render('view-request', [
            'model' => $model,
        ]);
    }

    /**
     * Обновление статуса заявки
     */
    public function actionUpdateRequestStatus()
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
        
        $requestId = Yii::$app->request->post('id');
        $status = Yii::$app->request->post('status');
        $comment = Yii::$app->request->post('comment', '');
        
        $model = Request::findOne($requestId);
        
        if ($model) {
            $model->status = $status;
            $model->admin_comment = $comment;
            
            if ($model->save()) {
                return [
                    'success' => true,
                    'message' => 'Статус заявки обновлен',
                ];
            }
        }
        
        return [
            'success' => false,
            'message' => 'Ошибка при обновлении статуса',
        ];
    }

    /**
     * Управление пользователями
     */
    public function actionUsers()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find()->orderBy(['created_at' => SORT_DESC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('users', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Редактирование пользователя
     */
    public function actionEditUser($id)
    {
        $model = User::findOne($id);
        
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Пользователь не найден');
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Данные пользователя обновлены');
            return $this->redirect(['users']);
        }

        return $this->render('edit-user', [
            'model' => $model,
        ]);
    }

    /**
     * Управление играми
     */
    public function actionGames()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Game::find()->orderBy(['name' => SORT_ASC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('games', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Добавление игры
     */
    public function actionCreateGame()
    {
        $model = new Game();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Игра успешно добавлена');
            return $this->redirect(['games']);
        }

        return $this->render('create-game', [
            'model' => $model,
        ]);
    }

    /**
     * Редактирование игры
     */
    public function actionEditGame($id)
    {
        $model = Game::findOne($id);
        
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Игра не найдена');
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Игра обновлена');
            return $this->redirect(['games']);
        }

        return $this->render('edit-game', [
            'model' => $model,
        ]);
    }

    /**
     * Управление игровыми сессиями
     */
    public function actionSessions()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => GameSession::find()
                ->with('game')
                ->orderBy(['date' => SORT_ASC, 'time' => SORT_ASC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('sessions', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Создание игровой сессии
     */
    public function actionCreateSession()
    {
        $model = new GameSession();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Игровая сессия создана');
            return $this->redirect(['sessions']);
        }

        $games = Game::find()
            ->select(['name', 'id'])
            ->indexBy('id')
            ->column();

        return $this->render('create-session', [
            'model' => $model,
            'games' => $games,
        ]);
    }

    /**
     * Редактирование игровой сессии
     */
    public function actionEditSession($id)
    {
        $model = GameSession::findOne($id);
        
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Сессия не найдена');
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Сессия обновлена');
            return $this->redirect(['sessions']);
        }

        $games = Game::find()
            ->select(['name', 'id'])
            ->indexBy('id')
            ->column();

        return $this->render('edit-session', [
            'model' => $model,
            'games' => $games,
        ]);
    }

    /**
     * Управление отзывами
     */
    public function actionReviews()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Review::find()
                ->with(['user', 'session.game'])
                ->orderBy(['created_at' => SORT_DESC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('reviews', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Одобрение отзыва
     */
    public function actionApproveReview($id)
    {
        $model = Review::findOne($id);
        
        if ($model) {
            $model->status = Review::STATUS_APPROVED;
            $model->save();
            Yii::$app->session->setFlash('success', 'Отзыв опубликован');
        }

        return $this->redirect(['reviews']);
    }

    /**
     * Отклонение отзыва
     */
    public function actionRejectReview($id)
    {
        $model = Review::findOne($id);
        
        if ($model) {
            $model->status = Review::STATUS_REJECTED;
            $model->save();
            Yii::$app->session->setFlash('success', 'Отзыв отклонен');
        }

        return $this->redirect(['reviews']);
    }

    /**
     * Статистика по играм
     */
    public function actionStats()
    {
        // Популярные игры
        $popularGames = Game::find()
            ->select(['games.*', 'COUNT(requests.id) as request_count'])
            ->leftJoin('game_sessions', 'games.id = game_sessions.game_id')
            ->leftJoin('requests', 'game_sessions.id = requests.session_id')
            ->groupBy('games.id')
            ->orderBy(['request_count' => SORT_DESC])
            ->limit(10)
            ->all();

        // Активность пользователей
        $activeUsers = User::find()
            ->select(['users.*', 'COUNT(requests.id) as request_count'])
            ->leftJoin('requests', 'users.id = requests.user_id')
            ->groupBy('users.id')
            ->orderBy(['request_count' => SORT_DESC])
            ->limit(10)
            ->all();

        // Заявки по дням
        $requestsByDay = Request::find()
            ->select(['DATE(FROM_UNIXTIME(created_at)) as date', 'COUNT(*) as count'])
            ->groupBy(['date'])
            ->orderBy(['date' => SORT_DESC])
            ->limit(30)
            ->asArray()
            ->all();

        return $this->render('stats', [
            'popularGames' => $popularGames,
            'activeUsers' => $activeUsers,
            'requestsByDay' => $requestsByDay,
        ]);
    }
}