<?php
use yii\helpers\Html;
use yii\helpers\Url;
use yii\widgets\ActiveForm;
use app\models\Request;
$this->title = 'Запись на игру';
?>

<div class="request-page">
    <div class="request-header">
        <div class="container">
            <h1 class="request-title">Запись на игру</h1>
            <p class="request-subtitle">Забронируйте места и присоединяйтесь к игровой сессии</p>
        </div>
    </div>

    <div class="container">
        <div class="request-grid">
            <!-- Форма записи -->
            <div class="request-form-container">
                <div class="form-card">
                    <div class="form-card-header">
                        <i class="fas fa-ticket-alt"></i>
                        <h2>Данные бронирования</h2>
                    </div>
                    
                    <?php $form = ActiveForm::begin([
                        'id' => 'request-form',
                        'options' => ['class' => 'request-form']
                    ]); ?>
                    
                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-gamepad"></i>
                            Выберите игру
                        </label>
                        <?= $form->field($model, 'session_id')->dropDownList(
                            \yii\helpers\ArrayHelper::map($sessions, 'id', function($session) {
                                return $session->game->name . ' - ' . date('d.m.Y', strtotime($session->date)) . ' в ' . date('H:i', strtotime($session->time));
                            }),
                            [
                                'prompt' => '— Выберите игру и время —',
                                'class' => 'form-select',
                                'id' => 'session-select'
                            ]
                        )->label(false) ?>
                        <div id="session-details" class="session-details" style="display: none;">
                            <!-- Сюда будет подгружаться информация о выбранной сессии -->
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label class="form-label">
                                <i class="fas fa-users"></i>
                                Количество игроков
                            </label>
                            <?= $form->field($model, 'player_count')->dropDownList([1 => '1 игрок', 2 => '2 игрока', 3 => '3 игрока', 4 => '4 игрока', 5 => '5 игроков'], [
                                'class' => 'form-select',
                                'id' => 'player-count'
                            ])->label(false) ?>
                        </div>
                        
                        <div class="form-group col-md-6">
                            <label class="form-label">
                                <i class="fas fa-money-bill"></i>
                                Способ оплаты
                            </label>
                            <?= $form->field($model, 'payment_method')->dropDownList(
                                Request::getPaymentMethods(),
                                [
                                    'prompt' => '— Выберите способ —',
                                    'class' => 'form-select'
                                ]
                            )->label(false) ?>
                        </div>
                    </div>
                    
                    <!-- Личные данные пользователя (только для просмотра) -->
                    <div class="user-info-card">
                        <h3><i class="fas fa-user-circle"></i> Ваши данные</h3>
                        <div class="user-info-grid">
                            <div class="user-info-item">
                                <span class="info-label">ФИО:</span>
                                <span class="info-value"><?= Html::encode($user->full_name) ?></span>
                            </div>
                            <div class="user-info-item">
                                <span class="info-label">Телефон:</span>
                                <span class="info-value"><?= Html::encode($user->phone) ?></span>
                            </div>
                            <div class="user-info-item">
                                <span class="info-label">Email:</span>
                                <span class="info-value"><?= Html::encode($user->email) ?></span>
                            </div>
                            <div class="user-info-item">
                                <span class="info-label">Статус:</span>
                                <span class="info-value status-badge"><?= $user->getPlayerStatusLabel() ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Итоговая стоимость -->
                    <div class="total-price-card">
                        <div class="total-price-label">
                            <i class="fas fa-calculator"></i>
                            <span>Итого к оплате:</span>
                        </div>
                        <div class="total-price-amount">
                            <span id="total-price">0</span> ₽
                        </div>
                    </div>
                    
                    <div class="form-check">
                        <?= $form->field($model, 'consent_data')->checkbox([
                            'template' => "{input} {label}",
                            'class' => 'checkbox',
                            'id' => 'consent_data',
                            'checked' => true
                        ]) ?>
                        <label for="consent_data" class="checkbox-label">
                            Я согласен на <a href="#" onclick="openConsentModal(event)">обработку персональных данных</a>
                        </label>
                    </div>
                    
                    <div class="form-actions">
                        <?= Html::submitButton('<i class="fas fa-check-circle"></i> Забронировать место', [
                            'class' => 'btn btn-primary btn-large'
                        ]) ?>
                        <a href="<?= Url::to(['site/schedule']) ?>" class="btn btn-outline">
                            <i class="fas fa-calendar-alt"></i>
                            Смотреть расписание
                        </a>
                    </div>
                    
                    <?php ActiveForm::end(); ?>
                </div>
            </div>
            
            <!-- Информационный сайдбар -->
            <div class="request-sidebar">
                <div class="info-card">
                    <i class="fas fa-info-circle"></i>
                    <h3>Как это работает?</h3>
                    <ol class="info-steps">
                        <li>Выберите игру и удобное время</li>
                        <li>Укажите количество игроков</li>
                        <li>Подтвердите бронирование</li>
                        <li>Дождитесь подтверждения от администратора</li>
                        <li>Приходите и играйте!</li>
                    </ol>
                </div>
                
                <div class="info-card warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <h3>Важно!</h3>
                    <ul class="warning-list">
                        <li>Отмена брони возможна не позднее чем за 24 часа</li>
                        <li>При опоздании более чем на 15 минут бронь аннулируется</li>
                        <li>С собой иметь хорошее настроение :)</li>
                    </ul>
                </div>
                
                <div class="info-card contact">
                    <i class="fas fa-headset"></i>
                    <h3>Есть вопросы?</h3>
                    <p>Свяжитесь с нами любым удобным способом</p>
                    <div class="contact-links">
                        <a href="tel:89999999999" class="contact-link">
                            <i class="fas fa-phone"></i>
                            8(999)999-99-99
                        </a>
                        <a href="mailto:info@nastolka.ru" class="contact-link">
                            <i class="fas fa-envelope"></i>
                            info@nastolka.ru
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Модальное окно с согласием на обработку данных -->
<div id="consentModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Согласие на обработку персональных данных</h3>
            <button class="modal-close" onclick="closeConsentModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div class="consent-text">
                <p>Нажимая кнопку «Забронировать место», я даю свое согласие ООО «Настолка» (далее — Оператор) на обработку моих персональных данных на следующих условиях:</p>
                
                <h4>1. Состав персональных данных</h4>
                <p>Фамилия, имя, отчество, номер контактного телефона, адрес электронной почты.</p>
                
                <h4>2. Цели обработки</h4>
                <p>Оформление и исполнение договора бронирования, связь с пользователем, направление уведомлений.</p>
                
                <h4>3. Действия с персональными данными</h4>
                <p>Сбор, запись, систематизация, накопление, хранение, уточнение, использование, передача, обезличивание, блокирование, удаление.</p>
                
                <h4>4. Срок действия согласия</h4>
                <p>Согласие действует со дня его подписания до дня отзыва в письменной форме.</p>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-primary" onclick="closeConsentModal()">Я ознакомлен(а)</button>
        </div>
    </div>
</div>

<style>
.request-page {
    padding-bottom: 60px;
}

.request-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 60px 0;
    margin-bottom: 40px;
    border-radius: 0 0 50px 50px;
    color: white;
    text-align: center;
}

.request-title {
    font-size: 42px;
    font-weight: 800;
    margin-bottom: 15px;
}

.request-subtitle {
    font-size: 18px;
    opacity: 0.95;
    max-width: 600px;
    margin: 0 auto;
}

.request-grid {
    display: grid;
    grid-template-columns: 1fr 380px;
    gap: 30px;
}

/* Форма */
.form-card {
    background: white;
    border-radius: 20px;
    box-shadow: 0 10px 40px rgba(0,0,0,0.05);
    overflow: hidden;
}

.form-card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 25px 30px;
    color: white;
    display: flex;
    align-items: center;
    gap: 15px;
}

.form-card-header i {
    font-size: 28px;
}

.form-card-header h2 {
    margin: 0;
    font-size: 22px;
    font-weight: 600;
}

.request-form {
    padding: 30px;
}

.form-group {
    margin-bottom: 25px;
}

.form-label {
    display: block;
    margin-bottom: 10px;
    font-weight: 600;
    color: #333;
    font-size: 15px;
}

.form-label i {
    color: #667eea;
    margin-right: 8px;
}

.form-control, .form-select {
    width: 100%;
    padding: 14px 18px;
    border: 2px solid #e0e0e0;
    border-radius: 12px;
    font-size: 15px;
    font-family: 'Montserrat', sans-serif;
    transition: all 0.3s;
    background: #f8f9fa;
}

.form-control:focus, .form-select:focus {
    border-color: #667eea;
    background: white;
    box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1);
    outline: none;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

/* Информация о пользователе */
.user-info-card {
    background: linear-gradient(135deg, #667eea08 0%, #764ba208 100%);
    border-radius: 16px;
    padding: 25px;
    margin-bottom: 25px;
    border: 1px solid rgba(102, 126, 234, 0.2);
}

.user-info-card h3 {
    margin: 0 0 20px;
    font-size: 18px;
    color: #667eea;
    display: flex;
    align-items: center;
    gap: 10px;
}

.user-info-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 20px;
}

.user-info-item {
    display: flex;
    flex-direction: column;
    gap: 5px;
}

.info-label {
    font-size: 13px;
    color: #666;
}

.info-value {
    font-size: 15px;
    font-weight: 600;
    color: #333;
}

.status-badge {
    display: inline-block;
    padding: 5px 12px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 50px;
    font-size: 12px;
    font-weight: 600;
}

/* Итоговая стоимость */
.total-price-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 16px;
    padding: 20px 25px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
    color: white;
}

.total-price-label {
    display: flex;
    align-items: center;
    gap: 10px;
    font-size: 18px;
    font-weight: 500;
}

.total-price-amount {
    font-size: 32px;
    font-weight: 800;
}

/* Чекбокс */
.form-check {
    display: flex;
    align-items: flex-start;
    gap: 12px;
    margin-bottom: 25px;
}

.checkbox {
    width: 20px;
    height: 20px;
    margin-top: 2px;
    cursor: pointer;
    accent-color: #667eea;
}

.checkbox-label {
    font-size: 14px;
    color: #666;
    line-height: 1.5;
    cursor: pointer;
}

.checkbox-label a {
    color: #667eea;
    text-decoration: none;
    font-weight: 500;
}

.checkbox-label a:hover {
    text-decoration: underline;
}

/* Кнопки */
.form-actions {
    display: flex;
    gap: 15px;
}

.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    padding: 14px 28px;
    font-size: 16px;
    font-weight: 600;
    text-decoration: none;
    border-radius: 12px;
    border: none;
    cursor: pointer;
    transition: all 0.3s;
    font-family: 'Montserrat', sans-serif;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    flex: 1;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 10px 30px rgba(102, 126, 234, 0.4);
}

.btn-outline {
    background: transparent;
    border: 2px solid #667eea;
    color: #667eea;
}

.btn-outline:hover {
    background: #667eea;
    color: white;
}

.btn-large {
    padding: 16px 32px;
    font-size: 18px;
}

/* Сайдбар */
.request-sidebar {
    display: flex;
    flex-direction: column;
    gap: 25px;
}

.info-card {
    background: white;
    border-radius: 20px;
    padding: 25px;
    box-shadow: 0 10px 40px rgba(0,0,0,0.05);
}

.info-card i {
    font-size: 32px;
    color: #667eea;
    margin-bottom: 15px;
}

.info-card h3 {
    margin: 0 0 15px;
    font-size: 18px;
    color: #333;
}

.info-card.warning {
    background: linear-gradient(135deg, #fff3cd 0%, #ffe69b 100%);
    border: 1px solid #ffe69b;
}

.info-card.warning i {
    color: #856404;
}

.info-card.warning h3 {
    color: #856404;
}

.info-card.contact {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
}

.info-card.contact i {
    color: #155724;
}

.info-card.contact h3 {
    color: #155724;
}

.info-steps, .warning-list {
    margin: 0;
    padding-left: 20px;
}

.info-steps li, .warning-list li {
    margin-bottom: 10px;
    color: #666;
    line-height: 1.5;
}

.contact-links {
    display: flex;
    flex-direction: column;
    gap: 12px;
    margin-top: 15px;
}

.contact-link {
    display: flex;
    align-items: center;
    gap: 10px;
    color: #155724;
    text-decoration: none;
    padding: 10px;
    background: rgba(255,255,255,0.5);
    border-radius: 10px;
    transition: all 0.3s;
}

.contact-link:hover {
    background: white;
}

/* Модальное окно */
.modal {
    display: none;
    position: fixed;
    z-index: 9999;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
    align-items: center;
    justify-content: center;
}

.modal.show {
    display: flex;
}

.modal-content {
    background: white;
    width: 90%;
    max-width: 600px;
    border-radius: 20px;
    box-shadow: 0 25px 50px rgba(0,0,0,0.3);
    animation: modalSlideIn 0.3s;
}

@keyframes modalSlideIn {
    from {
        transform: translateY(-50px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

.modal-header {
    padding: 25px 30px;
    border-bottom: 1px solid #e0e0e0;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 20px 20px 0 0;
}

.modal-header h3 {
    margin: 0;
    font-size: 20px;
}

.modal-close {
    background: none;
    border: none;
    color: white;
    font-size: 28px;
    cursor: pointer;
    opacity: 0.8;
}

.modal-close:hover {
    opacity: 1;
}

.modal-body {
    padding: 30px;
    max-height: 70vh;
    overflow-y: auto;
}

.modal-footer {
    padding: 20px 30px;
    border-top: 1px solid #e0e0e0;
    text-align: right;
}

.consent-text h4 {
    color: #667eea;
    margin: 20px 0 10px;
}

.consent-text p {
    color: #666;
    line-height: 1.6;
    margin-bottom: 15px;
}

/* Адаптивность */
@media (max-width: 992px) {
    .request-grid {
        grid-template-columns: 1fr;
    }
    
    .request-title {
        font-size: 32px;
    }
}

@media (max-width: 768px) {
    .request-header {
        padding: 40px 0;
    }
    
    .request-title {
        font-size: 28px;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .user-info-grid {
        grid-template-columns: 1fr;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .request-form {
        padding: 20px;
    }
}

@media (max-width: 480px) {
    .request-header {
        padding: 30px 0;
    }
    
    .request-title {
        font-size: 24px;
    }
    
    .total-price-label {
        font-size: 16px;
    }
    
    .total-price-amount {
        font-size: 24px;
    }
}
</style>

<script>
// Обновление цены при выборе сессии и количества игроков
document.addEventListener('DOMContentLoaded', function() {
    const sessionSelect = document.getElementById('session-select');
    const playerCount = document.getElementById('player-count');
    const totalPriceSpan = document.getElementById('total-price');
    
    // Данные о сессиях (нужно передать из контроллера)
    const sessions = <?= json_encode(\yii\helpers\ArrayHelper::map($sessions, 'id', function($session) {
        return [
            'price' => $session->price,
            'game' => $session->game->name,
            'date' => date('d.m.Y', strtotime($session->date)),
            'time' => date('H:i', strtotime($session->time)),
            'available' => $session->availableSlots
        ];
    })) ?>;
    
    function updateTotalPrice() {
        const sessionId = sessionSelect.value;
        const players = playerCount.value;
        
        if (sessionId && players && sessions[sessionId]) {
            const price = sessions[sessionId].price * players;
            totalPriceSpan.textContent = price.toLocaleString('ru-RU');
        } else {
            totalPriceSpan.textContent = '0';
        }
    }
    
    sessionSelect.addEventListener('change', updateTotalPrice);
    playerCount.addEventListener('change', updateTotalPrice);
    
    // Показываем детали сессии при выборе
    sessionSelect.addEventListener('change', function() {
        const sessionId = this.value;
        const detailsDiv = document.getElementById('session-details');
        
        if (sessionId && sessions[sessionId]) {
            const session = sessions[sessionId];
            detailsDiv.innerHTML = `
                <div class="session-info">
                    <div><i class="fas fa-gamepad"></i> ${session.game}</div>
                    <div><i class="fas fa-calendar"></i> ${session.date}</div>
                    <div><i class="fas fa-clock"></i> ${session.time}</div>
                    <div><i class="fas fa-tag"></i> ${session.price} ₽/чел</div>
                    <div><i class="fas fa-users"></i> Свободно: ${session.available} мест</div>
                </div>
            `;
            detailsDiv.style.display = 'block';
        } else {
            detailsDiv.style.display = 'none';
        }
    });
});

// Модальное окно
function openConsentModal(e) {
    e.preventDefault();
    document.getElementById('consentModal').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeConsentModal() {
    document.getElementById('consentModal').classList.remove('show');
    document.body.style.overflow = '';
}

// Закрытие по Escape
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeConsentModal();
    }
});

// Закрытие по клику вне модального окна
window.addEventListener('click', function(e) {
    const modal = document.getElementById('consentModal');
    if (e.target === modal) {
        closeConsentModal();
    }
});
</script>